<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\OpeningSchedule;
use App\Models\Store;
use App\Services\Bot\telegram;
use App\Traits\Upload;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class StoreController extends Controller
{
    use Upload;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function list(Request $request)
    {
        $search = $request->all();
        $dateSearch = Carbon::parse($request->created_at);

        $data['stores'] = Store::own($this->user->id)
            ->when(isset($search['name']), function ($query) use ($search) {
                return $query->where('name', 'LIKE', $search['name']);
            })
            ->when(isset($search['created_at']), function ($query) use ($dateSearch) {
                return $query->whereDate("created_at", $dateSearch);
            })
            ->when(isset($search['status']), function ($query) use ($search) {
                return $query->where('status', $search['status']);
            })
            ->orderBy('id', 'desc')->paginate(basicControl()->paginate);
        return view($this->theme . 'user.store.list', $data);
    }

    public function create()
    {
        return view($this->theme . 'user.store.create');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'shipping_charge' => 'required',
            'delivery_note' => 'required',
            'status' => 'required',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();
        try {
            $store = new Store();
            $store->user_id = auth()->id();
            $store->name = $request->name;
            $store->short_description = $request->short_description;
            $store->shipping_charge = $request->shipping_charge;
            $store->delivery_note = $request->delivery_note;
            $store->cash_on_delivery = $request->cash_on_delivery;
            $store->status = $request->status;
            $store->save();

            $data = $store->id . '|' . $store->name;
            $store->link = $this->encrypt($data);
            $command = config('command');
            $store->bot_command = $command;
            $store->save();
            DB::commit();
            return redirect()->back()->with('success', 'Store Create Successfully');
        }catch (\Exception $exception){
            DB::rollBack();
            return back()->with('error', 'something went wrong');
        }
    }

    public function edit($id)
    {
        $data['store'] = Store::own($this->user->id)->findOrFail($id);
        return view($this->theme . 'user.store.edit', $data);
    }

    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'shipping_charge' => 'required',
            'delivery_note' => 'required',
            'status' => 'required',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            $store = Store::own($this->user->id)->findOrFail($id);
            $store->name = $request->name;
            $store->short_description = $request->short_description;
            $store->shipping_charge = $request->shipping_charge;
            $store->delivery_note = $request->delivery_note;
            $store->cash_on_delivery = $request->cash_on_delivery;
            $store->status = $request->status;
            $store->save();
            return redirect()->back()->with('success', 'Store Updated Successfully');
        }catch (\Exception $exception){
            return back()->with('error', 'something went wrong');
        }
    }

    public function delete($id)
    {
        $store = Store::own($this->user->id)->with(['productsMap', 'orders'])->findOrFail($id);
        if (count($store->productsMap) > 0) {
            return back()->with('alert', 'This store has lot of menu items');
        }
        if (count($store->orders) > 0) {
            return back()->with('alert', 'This store has lot of orders');
        }
        $store->delete();
        return back()->with('success', 'Store Deleted Successfully');
    }

    public function storeQRCode($storeId)
    {
        $store = Store::own($this->user->id)->findOrFail($storeId);
        $qrCodeMSg = getSectionData('qr_code_hint','single');
        return view($this->theme . 'user.store.bot.qrCode', compact('store', 'qrCodeMSg'));
    }

    public function schedules($storeId = null)
    {
        $data['stores'] = Store::own($this->user->id)->where('status', 1)->get();

        if (count($data['stores']) < 1) {
            return back()->with('error', 'Please Create Store First');
        }

        if ($storeId == null) {
            $storeId = $data['stores'][0]->id;
        } else {
            Store::own($this->user->id)->where('status', 1)->findOrFail($storeId);
        }
        $data['times'] = OpeningSchedule::own($this->user->id)->where('store_id', $storeId)->get();
        if (count($data['times']) < 6) {
            $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
            for ($i = 0; $i < 7; $i++) {
                OpeningSchedule::firstOrCreate([
                    'user_id' => $this->user->id,
                    'store_id' => $storeId,
                    'day' => $days[$i],
                    'from' => Carbon::now()->toTimeString(),
                    'to' => Carbon::now()->toTimeString(),
                ]);
            }
            $data['times'] = OpeningSchedule::own($this->user->id)->where('store_id', $storeId)->get();
        }
        foreach ($data['times'] as $time) {
            $day[$time->day] = $time;
        }
        return view($this->theme . 'user.store.opining_schedule.index', $data, compact('day', 'storeId'));
    }

    public function scheduleUpdate(Request $request, $storeId)
    {
        $validator = Validator::make($request->all(), [
            'from.*' => 'required',
            'to.*' => 'required',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $requestData = (object)$request->all();
        if (!isset($requestData->offday)) {
            $requestData->offday = [];
        }

        $store = Store::own($this->user->id)->where('status', 1)->findOrFail($storeId);
        OpeningSchedule::own($this->user->id)->where('store_id', $store->id)->get()->map(function ($query, $i) use ($requestData) {
            $query->from = $requestData->from[$i];
            $query->to = $requestData->to[$i];
            if (in_array($query->day, $requestData->offday)) {
                $query->off_day = 1;
            }else{
                $query->off_day = 0;
            }
            $query->save();
        });
        return back()->with('success', 'Updated Successfully');
    }

    public function encrypt($data)
    {
        return implode(unpack("H*", $data));
    }

    public function telegramBot(Request $request, $storeId)
    {
        $store = Store::own($this->user->id)->findOrFail($storeId);
        if ($request->method() == 'GET') {
            return view($this->theme . 'user.store.bot.telegramBot', compact('store'));
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'telegram_token' => 'required',
                'telegram_bot_image' => 'sometimes|image|mimes:jpg,png,jpeg',
                'telegram_initiate_message' => 'required|string|min:5|max:500',
            ]);
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            try {
                if ($request->hasFile('telegram_bot_image')) {
                    $image = $this->fileUpload($request->telegram_bot_image, config('filelocation.telegramBot.path'), null, config('filelocation.telegramBot.size'), 'webp', 99);
                    if ($image){
                        $store->telegram_bot_image = $image['path'];
                        $store->telegram_bot_image_driver = $image['driver'];
                    }
                    throw_if(empty($image['path']), 'Image could not be uploaded.');
                }

                $store->telegram_token = $request->telegram_token;
                $store->telegram_initiate_message = $request->telegram_initiate_message;
                $store->save();

                $telegram = new telegram();
                $response = $telegram->getBot($request->telegram_token);

                if ($response['status'] == 'success') {
                    $store->telegram_id = $response['res']->result->id;
                    $store->telegram_first_name = $response['res']->result->first_name;
                    $store->telegram_username = $response['res']->result->username;
                    $store->save();
                }
                $wb_response = $telegram->webhookSet($request->telegram_token, $store->link);

                if ($wb_response['status'] == 'success') {
                    return back()->with('success', 'Bot has been created');
                }
                return back()->with('error', 'Something went wrong please try again');
            } catch (\Exception $e) {
                return back()->with('error', 'Something went wrong');
            }
        }
    }

    public function telegramDocs()
    {
        return view($this->theme . 'layouts.telegram');
    }

    public function messengerBot(Request $request, $id)
    {
        $store = Store::own($this->user->id)->findOrFail($id);
        if ($request->method() == 'GET') {
            return view($this->theme . 'user.store.bot.messengerBot', compact('store'));
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'messenger_token' => 'required|min:10',
                'messenger_bot_image' => 'sometimes|image|mimes:jpg,png,jpeg',
                'messenger_initiate_message' => 'required|string|min:5|max:500',
            ]);
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            try {
                if ($request->hasFile('messenger_bot_image')) {
                    $image = $this->fileUpload($request->messenger_bot_image, config('filelocation.messengerBot.path'), null, config('filelocation.messengerBot.size'), 'webp', 99);
                    if ($image){
                        $store->messenger_bot_image = $image['path'];
                        $store->messenger_bot_image_driver = $image['driver'];
                    }
                    throw_if(empty($image['path']), 'Image could not be uploaded.');
                }
                $store->messenger_token = $request->messenger_token;
                $store->messenger_initiate_message = $request->messenger_initiate_message;
                $store->save();

                $facebookObj = 'App\\Services\\FacebookServices';
                $data = $facebookObj::getId($store);
                if ($data['status'] == 'success') {
                    $store->messenger_id = $data['data'];
                }
                $store->save();
                return back()->with('success', 'Bot has been created');
            } catch (\Exception $e) {
                return back()->with('error', 'Something went wrong');
            }
        }
    }

    public function messengerDocs()
    {
        return view($this->theme . 'layouts.messenger');
    }

    public function whatsappBot(Request $request, $id)
    {
        $store = Store::own($this->user->id)->findOrFail($id);
        if ($request->method() == 'GET') {
            return view($this->theme . 'user.store.bot.whatsappBot', compact('store'));
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'whatsapp_sid' => 'required|min:10',
                'whatsapp_token' => 'required|min:10',
                'whatsapp_bot_image' => 'sometimes|image|mimes:jpg,png,jpeg',
                'whatsapp_initiate_message' => 'required|string|min:5|max:500',
                'whatsapp_business_account' => 'required|string|min:5|max:500',
            ]);
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            try {
                if ($request->hasFile('whatsapp_bot_image')) {
                    $image = $this->fileUpload($request->whatsapp_bot_image, config('filelocation.whatsappBot.path'), null, config('filelocation.whatsappBot.size'), 'webp', 99);
                    if ($image){
                        $store->whatsapp_bot_image = $image['path'];
                        $store->whatsapp_bot_image_driver = $image['driver'];
                    }
                    throw_if(empty($image['path']), 'Image could not be uploaded.');
                }

                $store->whatsapp_sid = $request->whatsapp_sid;
                $store->whatsapp_token = $request->whatsapp_token;
                $store->whatsapp_initiate_message = $request->whatsapp_initiate_message;
                $store->whatsapp_business_account = $request->whatsapp_business_account;
                $store->save();
                return back()->with('success', 'Bot has been created');
            } catch (\Exception $e) {
                return back()->with('error', 'Something went wrong');
            }
        }
    }

    public function whatsappDocs()
    {
        return view($this->theme . 'layouts.whatsapp');
    }

    public function commandBot($storeId)
    {
        $data['store'] = Store::own($this->user->id)->findOrFail($storeId);
        return view($this->theme . 'user.store.bot.command', $data);
    }

    public function commandBotEdit(Request $request, $storeId, $key)
    {
        $data['store'] = Store::own($this->user->id)->findOrFail($storeId);
        if ($request->method() == 'GET') {
            if ($data['store']->bot_command == null) {
                return back()->with('error', 'something went wrong');
            }
            $data['key'] = $key;
            return view($this->theme . 'user.store.bot.commandEdit', $data);
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'command_text' => 'required',
                'description' => 'required|string|min:5|max:200',
            ]);
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            $field = [];
            if ($data['store']->bot_command) {
                foreach ($data['store']->bot_command as $comkey => $command) {
                    if ($comkey == $key) {
                        $field[$comkey] = [
                            "key" => $command->key,
                            'text' => $request->command_text,
                            'description' => $request->description,
                        ];
                    } else {
                        $field[$comkey] = [
                            "key" => $command->key,
                            'text' => $command->text,
                            'description' => $command->description,
                        ];
                    }
                }
                $data['store']->bot_command = $field;
                $data['store']->save();
            }
            return back()->with('success', 'Command Updated Successfully');
        }
    }






}
