<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Yajra\DataTables\Facades\DataTables;

class OrderController extends Controller
{
    public function index($store_id = null)
    {
        $orderRecord = \Cache::get('orderRecord');
        if (!$orderRecord) {
            $orderRecord = Order::selectRaw('COUNT(id) AS totalOrder')
                ->selectRaw('COUNT(CASE WHEN stage = 1 THEN id END) AS deliveryDone')
                ->selectRaw('(COUNT(CASE WHEN stage = 1 THEN id END) / COUNT(id)) * 100 AS deliveryDonePercentage')
                ->selectRaw('COUNT(CASE WHEN stage = 2 THEN id END) AS inProcess')
                ->selectRaw('(COUNT(CASE WHEN stage = 2 THEN id END) / COUNT(id)) * 100 AS inProcessPercentage')
                ->selectRaw('COUNT(CASE WHEN stage = 3 THEN id END) AS outForDelivery')
                ->selectRaw('(COUNT(CASE WHEN stage = 3 THEN id END) / COUNT(id)) * 100 AS outForDeliveryPercentage')
                ->selectRaw('COUNT(CASE WHEN stage = 4 THEN id END) AS cancelOrder')
                ->selectRaw('(COUNT(CASE WHEN stage = 4 THEN id END) / COUNT(id)) * 100 AS cancelOrderPercentage')
                ->get()
                ->toArray();
            \Cache::put('orderRecord', $orderRecord);
        }

        $data['orderRecord'] = $orderRecord;
        $data['store_id'] = $store_id;
        return view('admin.order.index', $data);
    }

    public function search(Request $request, $store_id = null)
    {
        $search = $request->search['value'] ?? null;
        $filterInput = $request->filterInput;
        $filterStage = $request->filterStage;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $items = Order::query()
            ->with(['store', 'gateway'])
            ->orderBy('id', 'DESC')
            ->where('status', 1)
            ->when($store_id != 'null', function ($query) use ($store_id) {
                $query->where('store_id', $store_id);
            })
            ->when(!empty($search), function ($query) use ($search) {
                return $query->where('full_name', 'LIKE', "%$search%")
                    ->orWhere('email', 'LIKE', "%$search%")
                    ->orWhere('order_number', 'LIKE', "%$search%")
                    ->orWhere('total_amount', 'LIKE', "%$search%")
                    ->orWhereHas('store', function ($q2) use ($search) {
                        $q2->where('name', 'LIKE', "%$search%");
                    });
            })
            ->when(!empty($filterInput), function ($query) use ($filterInput) {
                return $query->where('full_name', 'LIKE', "%$filterInput%")
                    ->orWhere('email', 'LIKE', "%$filterInput%")
                    ->orWhere('order_number', 'LIKE', "%$filterInput%")
                    ->orWhere('total_amount', 'LIKE', "%$filterInput%")
                    ->orWhereHas('store', function ($q2) use ($filterInput) {
                        $q2->where('name', 'LIKE', "%$filterInput%");
                    });
            })
            ->when(!empty($filterStage), function ($query) use ($filterStage) {
                if ($filterStage == 'all') {
                    return $query->where('stage', '!=', null);
                }
                return $query->where('stage', $filterStage);
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            });

        return DataTables::of($items)
            ->addColumn('no', function ($item) {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('order_number', function ($item) {
                return $item->order_number;
            })
            ->addColumn('store', function ($item) {
                $storeUrl = route('admin.store.details', $item->store_id);
                return "<a href='" . $storeUrl . "' class='text-secondary'>" . optional($item->store)->name . "</a>";
            })
            ->addColumn('email', function ($item) {
                return $item->email;
            })
            ->addColumn('payment_amount', function ($item) {
                return currencyPosition(getAmount($item->payment_amount));
            })
            ->addColumn('payment_by', function ($item) {
                return $item->gateway_id == 0 ? 'Cash On Delivery' : $item->gateway->name;
            })
            ->addColumn('stage', function ($item) {
                $badgeClass = $item->stageClass($item->stage);
                $stage = $item->getStage($item->stage);
                return '<span class="badge bg-soft-' . $badgeClass . ' text-' . $badgeClass . '"><span class="legend-indicator bg-' . $badgeClass . '"></span>' . $stage . '</span>';
            })
            ->addColumn('date_time', function ($item) {
                return dateTime($item->created_at);
            })
            ->addColumn('action', function ($item) {
                return dateTime($item->created_at);
            })
            ->addColumn('action', function ($item) {
                $url = route('admin.order.details', $item->order_number);
                return "<a href='" . $url . "' class='btn btn-white btn-sm edit_btn'><i class='bi-eye fill me-1'></i> Details</a>";
            })
            ->rawColumns(['order_number', 'store', 'email', 'payment_amount', 'payment_by', 'stage', 'date_time', 'action'])
            ->make(true);
    }

    public function details($order_number)
    {
        $data['order'] = Order::with('shipping', 'details', 'details.product', 'store')->where('status', 1)->where('order_number', $order_number)->firstOrFail();
        return view('admin.order.details', $data);
    }
}
