<?php

namespace App\Services;

use App\Models\Order;
use App\Traits\Notify;
use Twilio\Rest\Client;

class WhatsappServices
{
	use Notify;

	public static function getMenu($store, $senderId)
	{
		try {
			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;
			$hi_text = $store->whatsapp_initiate_message ?? "Welcome please open menu and order food.";

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $hi_text . ' ' . route('public.view', [$senderId, $store->link]),
						"mediaUrl" => getFile($store->telegram_bot_image_driver, $store->whatsapp_bot_image),
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}

	public static function getLastOrder($store, $senderId)
	{
		try {
			$whatsappObj = new WhatsappServices();
			$order = $whatsappObj->lastOrder($senderId, $store);
			$order_text = "You have no order yet";
			if ($order) {
				$params = [
					'amount' => getAmount($order->payment_amount, 2),
					'orderNumber' => $order->order_number,
					'currency' => basicControl()->base_currency,
				];
				$order_text = $whatsappObj->botCommandText($store, 'lastorder', $params);
			}

			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $order_text,
						"mediaUrl" => route('downloadPdf', $order ? $order->id : '0'),
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}

	public static function getTotalOrders($store, $senderId)
	{
		try {
			$whatsappObj = new WhatsappServices();
			$orderCount = $whatsappObj->totalOrders($senderId, $store);
			$order_text = "You have no order yet";
			if ($orderCount) {
				$params = [
					'total' => $orderCount,
				];
				$order_text = $whatsappObj->botCommandText($store, 'totalorders', $params);
			}

			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $order_text,
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}

	public static function getLastOrderStatus($store, $senderId)
	{
		try {
			$whatsappObj = new WhatsappServices();
			$orderStatus = $whatsappObj->lastOrderStatus($senderId, $store);
			$order_text = "You have no order yet";
			if ($orderStatus) {
				$params = [
					'status' => $orderStatus,
				];
				$order_text = $whatsappObj->botCommandText($store, 'lastorderstatus', $params);
			}

			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $order_text,
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}

	public static function getTotalorderAmount($store, $senderId)
	{
		try {
			$whatsappObj = new WhatsappServices();
			$totalAmount = $whatsappObj->totalorderamount($senderId, $store);
			$order_text = "You have no order yet";
			if ($totalAmount) {
				$params = [
					'amount' => getAmount($totalAmount, 2),
					'currency' => basicControl()->base_currency,
				];
				$order_text = $whatsappObj->botCommandText($store, 'totalorderamount', $params);
			}

			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $order_text,
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}

	public static function getLast10Transaction($store, $senderId)
	{
		try {
			$order_text = "";
			$whatsappObj = new WhatsappServices();
			$orders = $whatsappObj->last10transaction($senderId, $store);
			if ($orders) {
				foreach ($orders as $order) {
					$params = [
						'orderNumber' => $order->order_number,
						'date' => dateTime($order->created_at, 'd/M/Y H:i'),
						'payAmount' => getAmount($order->payment_amount, 2),
						'currency' => basicControl()->base_currency,
					];
					$order_text .= $whatsappObj->botCommandText($store, 'last10transaction', $params) . PHP_EOL;
				}
			}

			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $order_text ?? 'you have no order yet',
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}

	public static function getHelp($store, $senderId)
	{
		try {
			$text = "/menu - To start the webpage and orders unlimited foods" . PHP_EOL . PHP_EOL;
			$commands = $store->bot_command;
			if ($commands) {
				foreach ($commands as $key => $command) {
					if ($key != 'orderconfirm') {
						$text .= "/" . $key . " - " . $command->description . PHP_EOL . PHP_EOL;
					}
				}
			}

			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $text ?? "No help found",
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}

	public static function unrecognized($store, $senderId)
	{
		try {
			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => 'Unrecognized command. Say what? /help for more details',
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}


	public static function orderConfirm($store, $senderId, $order)
	{
		try {
			$whatsappObj = new WhatsappServices();
			$params = [
				'orderNumber' => $order->order_number,
			];
			$order_text = $whatsappObj->botCommandText($store, 'orderconfirm', $params);

			$sid = $store->whatsapp_sid;
			$token = $store->whatsapp_token;
			$whatsAppNum = $store->whatsapp_business_account;

			$twilio = new Client($sid, $token);
			$message = $twilio->messages
				->create($senderId, // to
					[
						"from" => "whatsapp:" . $whatsAppNum,
						"body" => $order_text,
					]
				);
			return 0;
		} catch (\Exception $e) {
			return config('bot_error_msg.msg');
		}
	}


	public function lastOrder($senderId, $store)
	{
		$order = Order::where('store_id', $store->id)->where('status', 1)
			->where('sender_id', $senderId)
			->orderBy('id', 'desc')->first();
		return $order;
	}

	public function totalOrders($senderId, $store)
	{
		$order = Order::where('store_id', $store->id)->where('status', 1)
			->where('sender_id', $senderId)
			->orderBy('id', 'desc')->get();
		return count($order);
	}

	public function lastOrderStatus($senderId, $store)
	{
		$order = Order::where('store_id', $store->id)->where('status', 1)
			->where('sender_id', $senderId)
			->orderBy('id', 'desc')->first();
		if ($order) {
			switch ($order->stage) {
				case 0:
					return 'Arrived';
				case 1:
					return 'Delivery Completed';
				case 2:
					return 'In Process';
				case 3:
					return 'Out For Delivery';
				case 4:
					return 'Cancel';
			}

		}
	}

	public function totalorderamount($senderId, $store)
	{
		$order = Order::where('store_id', $store->id)->where('status', 1)
			->where('sender_id', $senderId)
			->sum('payment_amount');
		return $order;
	}

	public function last10transaction($senderId, $store)
	{
		$orders = Order::where('store_id', $store->id)->where('status', 1)
			->where('sender_id', $senderId)
			->orderBy('id', 'desc')->take(10)->get();
		return $orders;
	}
}
