<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Store;
use App\Models\StoreShipping;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class StoreShippingController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function index(Request $request)
    {
        $search = $request->all();
        $dateSearch = Carbon::parse($request->created_at);

        $data['stores'] = Store::own($this->user->id)->orderBy('name', 'asc')->toBase()->get();
        $data['shipping'] = StoreShipping::own($this->user->id)
            ->when(isset($search['store_name']), function ($query) use ($search) {
                $query->wherehas('store', function ($query) use ($search) {
                    $query->where('name', 'like', '%' . $search['store_name'] . '%');
                });
            })
            ->when(isset($search['created_at']), function ($query) use ($dateSearch) {
                return $query->whereDate("created_at", $dateSearch);
            })
            ->when(isset($search['address']), function ($query) use ($search) {
                $query->where('address', 'LIKE', '%' . $search['address'] . '%');
            })
            ->when(isset($search['status']), function ($query) use ($search) {
                $query->where('status', $search['status']);
            })
            ->orderBy('id', 'desc')->latest()->paginate(basicControl()->paginate);
        return view($this->theme . 'user.store.shipping.index', $data);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'store' => 'required',
            'address' => 'required',
            'charge' => 'required|min:0|not_in:0',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $storeShipping = new StoreShipping();
        $storeShipping->user_id = $this->user->id;
        $storeShipping->store_id = $request->store;
        $storeShipping->address = $request->address;
        $storeShipping->charge = $request->charge;
        $storeShipping->status = $request->status;

        $storeShipping->save();
        session()->flash('success', 'Created Successfully');
        return response()->json([
            'status' => 'success',
            'url' => route('user.store.shipping.index')
        ]);
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'store' => 'required',
            'address' => 'required',
            'charge' => 'required|min:0|not_in:0',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $storeShipping = StoreShipping::own($this->user->id)->find($request->id);
        $storeShipping->store_id = $request->store;
        $storeShipping->address = $request->address;
        $storeShipping->charge = $request->charge;
        $storeShipping->status = $request->status;

        $storeShipping->save();
        session()->flash('success', 'Updated Successfully');
        return response()->json([
            'status' => 'success',
            'url' => route('user.store.shipping.index')
        ]);
    }

    public function delete($id)
    {
        $storeShipping = StoreShipping::own($this->user->id)->with(['orders'])->findOrFail($id);
        if (count($storeShipping->orders) > 0) {
            return back()->with('error', 'This shipping has lot of orders');
        }
        $storeShipping->delete();
        return back()->with('success', 'Deleted Successfully');
    }
}
