<?php

namespace App\Http\Controllers;

use App\Models\ProductStoreMap;
use App\Models\Store;
use App\Models\StoreCategory;
use App\Models\StoreProduct;
use App\Models\StoreProductOptional;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;


class StoreShopController extends Controller
{
    public function __construct()
    {
        $this->theme = template();
    }

    public function shopProduct(Request $request, $senderId, $link, $category = null)
    {
        $clearSession = false;
        if ($category == 'clearSession') {
            $clearSession = true;
        }

        if ($category == 'telegram') {
            Session::put('bot_type', 'telegram');
        }
        $search = $request->all();
        $store = Store::where('link', $link)->with(['productsMap.product'])->where('status', 1)->firstOrFail();

        $productId = array();
        $categoryId = array();
        foreach ($store->productsMap as $productMap) {
            $productId[] = $productMap->product_id;
        }

        $data['products'] = StoreProduct::whereIn('id', $productId)->where('status', 1)
            ->when($category && $category != 'clearSession' && $category != 'telegram', function ($query) use ($category) {
                $query->where('category_id', $category);
            })
            ->when(isset($search['search']), function ($query) use ($search) {
                $query->where('name', 'LIKE', '%' . $search['search'] . '%')
                    ->orWhere('price', 'LIKE', '%' . $search['search'] . '%');
            })
            ->orderBy('id', 'desc')->get();

        foreach ($store->productsMap as $productMap) {
            $categoryId[] = optional($productMap->product)->category_id;
        }
        $categories = StoreCategory::whereIn('id', $categoryId)->get();
        $data['categories'] = $categories;

        $data['link'] = $link;
        Session::put('link', $link);
        return view($this->theme . 'user.shop.productList', $data, compact('senderId', 'store', 'clearSession'));
    }

    public function shopProductDetails($senderId, $link = null, $title, $id)
    {
        $data['link'] = $link;
        $store = Store::where('link', $link)->where('status', 1)->firstOrFail();
        $data['product'] = StoreProduct::where('status', 1)->findOrFail($id);

        $fromPrice = 0;
        if ($data['product']->variants) {
            foreach ($data['product']->variants as $key => $variant) {
                if ($key == 0) {
                    $fromPrice = $variant->variant_price;
                }
                if ($fromPrice > $variant->variant_price) {
                    $fromPrice = $variant->variant_price;
                }
            }
        }
        return view($this->theme . 'user.shop.productDetails', $data, compact('senderId', 'store', 'fromPrice'));
    }

    public function addCart(Request $request)
    {
        $isProductValid = ProductStoreMap::where('store_id', $request->storeId)->where('product_id', $request->productId)->exists();
        if (!$isProductValid) {
            Log::error('Product validation failed.', ['storeId' => $request->storeId, 'productId' => $request->productId]);
            return 0;
        }

        $product = StoreProduct::where('status', 1)->find($request->productId);
        if (!$product) {
            return 0;
        }
        $image = getFile($product->driver, $product->image);

        $extraProductPrice = 0;
        if (isset($request->extra_products)) {
            $extraProducts = StoreProductOptional::where('status', 1)->whereIn('id', $request->extra_products)->get();
            foreach ($extraProducts as $extraProduct) {
                $extraProductPrice += $extraProduct->price;
            }
        }

        if ($product->variants) {
            $quit = true;
            foreach ($product->variants as $variant) {
                if ($variant->variant_name == $request->variantName && $variant->variant_price == $request->variantPrice) {
                    $quit = false;
                }
            }
            if ($quit) {
                return 0;
            }
            $price = $request->variantPrice;
        } else {
            $price = $product->price;
        }
        $totalPrice = ($price + $extraProductPrice) * $request->counter;
        $cart = [
            'storeId' => $request->storeId,
            'productName' => $product->name,
            'productId' => $product->id,
            'price' => getAmount($price, 2),
            'totalPrice' => getAmount($totalPrice, 2),
            'quantity' => $request->counter,
            'image' => $image,
            'url' => route('public.product.update', [$request->senderId, $request->link, $product->name, $product->id]),
            'variantName' => $request->variantName ?? null,
            'variantPrice' => getAmount($request->variantPrice, 2) ?? null,
            'extra_products' => $request->extra_products ?? [],
        ];
        session()->flash('success', ucfirst($product->name) . ' Added to Cart');
        return response()->json([
            'status' => 'success',
            'data' => $cart,
            'url' => route('public.view', [$request->senderId, $request->link]),
        ]);
    }

    public function cartItem($senderId, $link)
    {
        $data['link'] = $link;
        $data['senderId'] = $senderId;
        return view($this->theme . 'user.shop.productCart', $data);
    }

    public function removeCart(Request $request)
    {
        if (isset($request->id)) {
            $cart = $request->items;
            if (isset($cart[$request->id])) {
                unset($cart[$request->id]);
                return response()->json([
                    'status' => 'success',
                    'item' => $cart,
                ]);
            }
        }
    }

    public function updateCart(Request $request)
    {
        if ($request->items) {
            $carts = $request->items;
            if (isset($carts[$request->id])) {
                if ($request->type == 'increment') {
                    $carts[$request->id]["totalPrice"] = getAmount($carts[$request->id]["totalPrice"] + ($carts[$request->id]["totalPrice"] / $carts[$request->id]["quantity"]), 2);
                    $carts[$request->id]["quantity"] = $carts[$request->id]["quantity"] + 1;
                } else {
                    $carts[$request->id]["totalPrice"] = getAmount($carts[$request->id]["totalPrice"] - ($carts[$request->id]["totalPrice"] / $carts[$request->id]["quantity"]), 2);
                    $carts[$request->id]["quantity"] = $carts[$request->id]["quantity"] - 1;
                }
                return response()->json([
                    'status' => 'success',
                    'item' => $carts,
                ]);
            }
        }
    }


    public function shopCartUpdate(Request $request, $senderId, $link = null, $title, $id)
    {
        $data['link'] = $link;
        $store = Store::where('link', $link)->where('status', 1)->firstOrFail();
        $data['product'] = StoreProduct::where('status', 1)->findOrFail($id);

        $fromPrice = 0;
        if ($data['product']->variants) {
            foreach ($data['product']->variants as $key => $variant) {
                if ($key == 0) {
                    $fromPrice = $variant->variant_price;
                }
                if ($fromPrice > $variant->variant_price) {
                    $fromPrice = $variant->variant_price;
                }
            }
        }
        return view($this->theme . 'user.shop.cartUpdate', $data, compact('senderId', 'store', 'fromPrice'));
    }

    public function updateCartDetails(Request $request)
    {
        $isProductValid = ProductStoreMap::where('store_id', $request->storeId)->where('product_id', $request->productId)->exists();
        if (!$isProductValid) {
            return 0;
        }

        $product = StoreProduct::where('status', 1)->find($request->productId);
        if (!$product) {
            return 0;
        }
        $image = getFile($product->driver, $product->image);

        $extraProductPrice = 0;
        if (isset($request->extra_products)) {
            $extraProducts = StoreProductOptional::where('status', 1)->whereIn('id', $request->extra_products)->get();
            foreach ($extraProducts as $extraProduct) {
                $extraProductPrice += $extraProduct->price;
            }
        }

        if ($product->variants) {
            $quit = true;
            foreach ($product->variants as $variant) {
                if ($variant->variant_name == $request->variantName && $variant->variant_price == $request->variantPrice) {
                    $quit = false;
                }
            }
            if ($quit) {
                return 0;
            }
            $price = $request->variantPrice;
        } else {
            $price = $product->price;
        }
        $totalPrice = ($price + $extraProductPrice) * $request->counter;

        $cart = $request->items;
        $cart[$request->id] = [
            'storeId' => $request->storeId,
            'productName' => $product->name,
            'productId' => $product->id,
            'price' => getAmount($price, 2),
            'totalPrice' => getAmount($totalPrice, 2),
            'quantity' => $request->counter,
            'image' => $image,
            'url' => route('public.product.update', [$request->senderId, $request->link, $product->name, $product->id]),
            'variantName' => $request->variantName ?? null,
            'variantPrice' => getAmount($request->variantPrice, 2) ?? null,
            'extra_products' => $request->extra_products ?? [],
        ];

        return response()->json([
            'status' => 'success',
            'item' => $cart,
            'url' => route('public.view', [$request->senderId, $request->link]),
        ]);
    }

}
