<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PurchasePlan;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Yajra\DataTables\Facades\DataTables;

class PurchasePlanController extends Controller
{
    public function index()
    {
        return view('admin.plan.purchase_list');
    }

    public function search(Request $request)
    {
        $search = $request->search['value'] ?? null;
        $filterInput = $request->filterInput;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $purchasePlan = PurchasePlan::query()
            ->with(['user'])
            ->orderBy('id', 'DESC')
            ->when(!empty($search), function ($query) use ($search) {
                return $query->where('plan_name', 'LIKE', "%$search%")
                    ->orWhereHas('user', function ($q2) use ($search) {
                    $q2->where('firstname', 'LIKE', "%$search%")
                        ->orWhere('lastname', 'LIKE', "%$search%")
                        ->orWhere('username', 'LIKE', "%$search%")
                        ->orWhere('email', 'LIKE', "%$search%");
                });
            })
            ->when(!empty($filterInput), function ($query) use ($filterInput) {
                return $query->where('plan_name', 'LIKE', "%$filterInput%")
                    ->orWhereHas('user', function ($q2) use ($filterInput) {
                    $q2->where('firstname', 'LIKE', "%$filterInput%")
                        ->orWhere('lastname', 'LIKE', "%$filterInput%")
                        ->orWhere('username', 'LIKE', "%$filterInput%")
                        ->orWhere('email', 'LIKE', "%$filterInput%");
                });
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            });

        return DataTables::of($purchasePlan)
            ->addColumn('no', function ($item) {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('user', function ($item) {
                $url = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="'.$url.'">
                            <div class="flex-shrink-0">
                              ' . $item->user->profilePicture() . '
                            </div>
                            <div class="flex-grow-1 ms-3">
                              <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                              <span class="fs-6 text-body">' . optional($item->user)->email . '</span>
                            </div>
                      </a>';
            })
            ->addColumn('plan-name', function ($item) {
                $url = route('admin.plan.edit', $item->plan_id);
                return '<a class="text-secondary" href="'.$url.'">'.trans($item->plan_name).'</a>';
            })
            ->addColumn('plan-price', function ($item) {
                return currencyPosition($item->price);
            })
            ->addColumn('schedule', function ($item) {
                return $item->schedule_name;
            })
            ->addColumn('purchased-at', function ($item) {
                return dateTime($item->purchase_date);
            })
            ->addColumn('expiry-date', function ($item) {
                $class = $item->expiry_date >= Carbon::now() ? 'text-success' : 'text-danger';
                return '<span class="'.$class.'">'.dateTime($item->expiry_date).'</span>';
            })
            ->addColumn('status', function ($item) {
                $badgeClass = $item->expiry_date >= Carbon::now() ? 'success text-success' : 'danger text-danger';
                $legendBgClass = $item->expiry_date >= Carbon::now() ? 'success' : 'danger';
                $status = $item->expiry_date >= Carbon::now() ? 'Active' : 'Expired';
                return '<span class="badge bg-soft-'.$badgeClass.'"><span class="legend-indicator bg-'.$legendBgClass.'"></span>'.$status.'</span>';
            })
            ->addColumn('action', function ($item) {
                return '<div class="btn-group" role="group">
                        <a href="javascript:void(0)" class="btn btn-white btn-sm viewDetails" data-plan="'.htmlspecialchars(json_encode($item), ENT_QUOTES, 'UTF-8').'" data-bs-target="#detailsModal" data-bs-toggle="modal">
                        <i class="bi-eye me-1"></i> ' . trans("Details") . '
                      </a>
                  </div>';
            })->rawColumns(['user', 'plan-name', 'plan-price', 'schedule', 'purchased-at', 'expiry-date', 'status', 'action'])
            ->make(true);
    }
}
